package org.msh.tb.bd.dashboard;

import org.jboss.seam.Component;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.faces.Redirect;
import org.msh.tb.bd.Quarter;
import org.msh.tb.entities.AdministrativeUnit;
import org.msh.tb.entities.UserWorkspace;
import org.msh.tb.entities.Workspace;
import org.msh.tb.indicators.core.IndicatorFilters;
import org.msh.tb.login.UserSession;
import org.msh.utils.HashUtils;
import org.msh.utils.date.DateUtils;
import org.msh.utils.date.Period;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;

/**
 * Created by mauri on 20/05/2017.
 */
@Name("dashboardIndicatorUtils")
public class DashboardIndicatorUtils {

    @In
    Redirect redirect;

    public static final String DASHBOARD_INDICATORS_PREFIX = "DASHIND";

    /**
     *
     * @param caseQtt
     * @param population
     * @return caseQtt*100000 / population. Return 0 if case qtt or population is zero or null.
     */
    public static Float calcPopulationRate(Long caseQtt, Long population) {
        if (caseQtt == null || caseQtt <= 0 || population == null || population <= 0) {
            return new Float(0);
        }

        BigDecimal bd = new BigDecimal((caseQtt*100000.0)/population).setScale(3, RoundingMode.HALF_EVEN);
        Float result = new Float(bd.floatValue());

        return result;
    }

    /**
     *
     * @param num1
     * @param num2
     * @return num1*100 / num2. Return 0 if case qtt or population is zero or null.
     */
    public static Float calcPercentage(Long num1, Long num2) {
        if (num1 == null || num1 <= 0 || num2 == null || num2 <= 0) {
            return new Float(0);
        }

        BigDecimal bd = new BigDecimal((num1*100.0)/num2).setScale(2, RoundingMode.HALF_EVEN);
        Float result = new Float(bd.floatValue());

        return result;
    }

    /**
     * @param userWorkspace
     * @return the hash corresponding to the userWorkspace configuration
     */
    public static String getHash(UserWorkspace userWorkspace) {
        AdministrativeUnit administrativeUnit = getAdminUnit(userWorkspace);

        if (administrativeUnit == null) {
            return getHash(userWorkspace.getWorkspace());
        } else {
            return getHash(administrativeUnit);
        }
    }

    /**
     * @param workspace
     * @return the hash for workspace level
     */
    public static String getHash(Workspace workspace) {
        return HashUtils.generateHash(DASHBOARD_INDICATORS_PREFIX + "." + workspace.getId());
    }

    /**
     * @param adminUnit
     * @return the hash for admin unit level
     */
    public static String getHash(AdministrativeUnit adminUnit) {
        return HashUtils.generateHash(DASHBOARD_INDICATORS_PREFIX + "." + adminUnit.getId());
    }

    /**
     * @return the dashboard admin unit according to userWorkspace. Will return null if user have national view.
     */
    public static AdministrativeUnit getAdminUnit(UserWorkspace userWorkspace) {
        switch (userWorkspace.getView()) {
            case TBUNIT:
                return getAdminUnitByLevel(userWorkspace.getTbunit().getAdminUnit(), 3);
            case ADMINUNIT:
                return userWorkspace.getAdminUnit();
            case COUNTRY:
                return null;
        }

        throw new RuntimeException("Could not check");
    }

    /**
     * @param adminUnit
     * @param level
     * @return the adminUnit based on its level
     */
    public static AdministrativeUnit getAdminUnitByLevel(AdministrativeUnit adminUnit, int level) {
        if (adminUnit == null) {
            return null;
        }

        if (adminUnit.getCountryStructure().getLevel() == level) {
            return adminUnit;
        }

        return getAdminUnitByLevel(adminUnit.getParent(), level);
    }

    public void initializeFilters(String indicatorCode) {
        IndicatorFilters filters = (IndicatorFilters) Component.getInstance("indicatorFilters");
        Period p;

        if (indicatorCode != null && (indicatorCode.equals("indicator07") || indicatorCode.equals("indicator11") || indicatorCode.equals("geodistribution"))) {
            p = getLastQuarterPeriod();
        } else {
            p = getLast12MonthsPeriod();
        }

        // initialize date filters
        filters.setIniMonth(DateUtils.monthOf(p.getIniDate()));
        filters.setIniYear(DateUtils.yearOf(p.getIniDate()));
        filters.setEndMonth(DateUtils.monthOf(p.getEndDate()));
        filters.setEndYear(DateUtils.yearOf(p.getEndDate()));

        // initialize admin unit selection filter
        AdministrativeUnit au = getAdminUnit(UserSession.getUserWorkspace());
        filters.getTbunitselection().getAuselection().setSelectedUnit(au);

        // redirect to indicator page
        redirect.setViewId("/custom/bd/dashboard/" + indicatorCode + ".xhtml");
        redirect.execute();
    }

    public static Period getLast12MonthsPeriod() {
        Date iniDate;
        Date endDate;
        Date auxDt;

        auxDt = DateUtils.incDays(DateUtils.getDate(), -365);
        iniDate = DateUtils.newDate(DateUtils.yearOf(auxDt),
                DateUtils.monthOf(auxDt),
                01);

        auxDt = DateUtils.incDays(DateUtils.getDate(), -30);
        endDate = DateUtils.newDate(DateUtils.yearOf(auxDt),
                DateUtils.monthOf(auxDt),
                DateUtils.daysInAMonth(DateUtils.yearOf(auxDt), DateUtils.monthOf(auxDt)));

        return new Period(iniDate, endDate);
    }

    public static Period getLastQuarterPeriod() {
        Quarter quarter = Quarter.getCurrentQuarter().getPreviousQuarter();
        return new Period(quarter.getIniDate(), quarter.getEndDate());
    }

    public static Integer getMaximumCircleSize(AdministrativeUnit au) {
        if (au == null) {
            return 60000;
        }

        if (au.getLevel() == 1) {
            return 20000;
        }

        if (au.getLevel() == 2) {
            return 15000;
        }

        return null;
    }

}
